/***************************************************************************
*   Copyright (C) 2004-2009 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <ntqfile.h>
#include <ntqstringlist.h>

#include <settings.h>
#include <element.h>
#include <fileexporterbibtex.h>
#include "fileexporterrtf.h"

namespace BibTeX
{

    FileExporterRTF::FileExporterRTF() : FileExporterToolchain(), m_latexLanguage( "english" ), m_latexBibStyle( "plain" )
    {
        laTeXFilename = TQString( workingDir ).append( "/bibtex-to-rtf.tex" );
        bibTeXFilename = TQString( workingDir ).append( "/bibtex-to-rtf.bib" );
        outputFilename = TQString( workingDir ).append( "/bibtex-to-rtf.rtf" );
    }

    FileExporterRTF::~FileExporterRTF()
    {
        // nothing
    }

    bool FileExporterRTF::save( TQIODevice* iodevice, const File* bibtexfile, TQStringList *errorLog )
    {
        m_mutex.lock();
        bool result = FALSE;

        TQFile bibtexFile( bibTeXFilename );
        if ( bibtexFile.open( IO_WriteOnly ) )
        {
            FileExporter * bibtexExporter = new FileExporterBibTeX();
            result = bibtexExporter->save( &bibtexFile, bibtexfile, errorLog );
            bibtexFile.close();
            delete bibtexExporter;
        }

        if ( result )
            result = generateRTF( iodevice, errorLog );

        m_mutex.unlock();
        return result;
    }

    bool FileExporterRTF::save( TQIODevice* iodevice, const Element* element, TQStringList *errorLog )
    {
        m_mutex.lock();
        bool result = FALSE;

        TQFile bibtexFile( bibTeXFilename );
        if ( bibtexFile.open( IO_WriteOnly ) )
        {
            FileExporter * bibtexExporter = new FileExporterBibTeX();
            result = bibtexExporter->save( &bibtexFile, element, errorLog );
            bibtexFile.close();
            delete bibtexExporter;
        }

        if ( result )
            result = generateRTF( iodevice, errorLog );

        m_mutex.unlock();
        return result;
    }

    void FileExporterRTF::setLaTeXLanguage( const TQString& language )
    {
        m_latexLanguage = language;
    }

    void FileExporterRTF::setLaTeXBibliographyStyle( const TQString& bibStyle )
    {
        m_latexBibStyle = bibStyle;
    }

    bool FileExporterRTF::generateRTF( TQIODevice* iodevice, TQStringList *errorLog )
    {
        TQStringList cmdLines = TQStringList::split( '|', "latex bibtex-to-rtf.tex|bibtex bibtex-to-rtf|latex bibtex-to-rtf.tex|latex2rtf bibtex-to-rtf.tex" );

        if ( writeLatexFile( laTeXFilename ) && runProcesses( cmdLines, errorLog ) && writeFileToIODevice( outputFilename, iodevice ) )
            return TRUE;
        else
            return FALSE;
    }

    bool FileExporterRTF::writeLatexFile( const TQString &filename )
    {
        TQFile latexFile( filename );
        if ( latexFile.open( IO_WriteOnly ) )
        {
            TQTextStream ts( &latexFile );
            ts.setEncoding( TQTextStream::UnicodeUTF8 );
            ts << "\\documentclass{article}\n";
            if ( kpsewhich( "t1enc.dfu" ) )
                ts << "\\usepackage[T1]{fontenc}\n";
            if ( kpsewhich( "babel.sty" ) )
                ts << "\\usepackage[" << m_latexLanguage << "]{babel}\n";
            if ( kpsewhich( "url.sty" ) )
                ts << "\\usepackage{url}\n";
            if ( m_latexBibStyle.startsWith( "apacite" ) && kpsewhich( "apacite.sty" ) )
                ts << "\\usepackage[bibnewpage]{apacite}\n";
            ts << "\\bibliographystyle{" << m_latexBibStyle << "}\n";
            ts << "\\begin{document}\n";
            ts << "\\nocite{*}\n";
            ts << "\\bibliography{bibtex-to-rtf}\n";
            ts << "\\end{document}\n";
            latexFile.close();
            return TRUE;
        }
        else
            return FALSE;

    }

}
