/***************************************************************************
                          batchwizard.cpp  -  description
                             -------------------
    begin                : Sun Mar 20 2005
    copyright            : (C) 2005 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "batchwizard.h"
#include "batchwizard.moc"
#include "barcodeprinterdlg.h"
#include "batchprinter.h"
#include "csvfile.h"
#include "definition.h"
#include "encodingcombo.h"
#include "printersettings.h"
#include "printlabeldlg.h"
#include "smalldialogs.h"
#include "sqltables.h"
#include "tokenprovider.h"
#include "xmlutils.h"

#include <tqbuffer.h>
#include <tqcheckbox.h>
#include <tqclipboard.h>
#include <tqcursor.h>
#include <tqdom.h>
#include <tqheader.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqradiobutton.h>
#include <tqsqlselectcursor.h> 
#include <tqtooltip.h>
#include <tqvbuttongroup.h>
#include <tqwidgetstack.h>

#include <tdeabc/addressee.h>
#include <tdeabc/addresseelist.h>
#include <tdeabc/addressbook.h>
#include <tdeabc/stdaddressbook.h>
#include <tdeapplication.h>
#include <kcombobox.h>
#include <kcompletion.h>
#include <tdefiledialog.h>
#include <kiconloader.h>
#include <kimageio.h>
#include <klineedit.h>
#include <tdelistbox.h>
#include <tdelistview.h>
#include <tdemessagebox.h>
#include <knuminput.h>
#include <tdepopupmenu.h>
#include <kpushbutton.h>
#include <kurlrequester.h>
#include <tqtable.h>
#include <tqvbox.h>

#define PNG_FORMAT "PNG"

class AddressListViewItem : public TDEListViewItem {
public:
    AddressListViewItem(TQListView *parent, TDEABC::Addressee & addr )
        : TDEListViewItem( parent ), m_address( addr )
        {
            this->setText( 0, m_address.givenName() );
            this->setText( 1, m_address.familyName() );
            this->setText( 2, m_address.preferredEmail() );
        }
    
    const TDEABC::Addressee & address() const {
        return m_address;
    }

private:
    TDEABC::Addressee m_address;

};

BatchWizard::BatchWizard( TQWidget* parent, const char* name )
    : DCOPObject("BatchPrinting"),
      KWizard( parent, name )
{
    setupPage1();
    setupPage2();
    setupPage3();
    setupPage4();
    setupPage5();
    setupPage10();

    compGroup = new TDECompletion();

    enableControls();
    setupSql();

    show();
}

BatchWizard::~BatchWizard()
{
    delete compGroup;
}

void BatchWizard::setupPage1()
{
    page1 = new TQWidget( this, "page1" );
    TQVBoxLayout* pageLayout = new TQVBoxLayout( page1, 11, 6, "pageLayout");

    TQLabel* label = new TQLabel( i18n("<qt>This wizard will guide you through the process "
				     "of printing many labels with KBarcode.<br>The first step "
				     "is to select the KBarcode label file you want to print.</qt>"), page1 );
    pageLayout->addWidget( label );
   
    m_url = new KURLRequester( page1 );
    m_url->setMode( KFile::File | KFile::ExistingOnly | KFile::LocalOnly );
    m_url->setFilter( "*.kbarcode" );

    label = new TQLabel( i18n("&Filename:"), page1 );
    label->setBuddy( m_url );

    TQSpacerItem* spacer = new TQSpacerItem( 20, 20, TQSizePolicy::Expanding, TQSizePolicy::Expanding );

    pageLayout->addWidget( label );
    pageLayout->addWidget( m_url );
    pageLayout->addItem( spacer );

    addPage( page1, i18n("File Selection") );

    connect( m_url, TQT_SIGNAL( textChanged( const TQString & ) ), this, TQT_SLOT( enableControls() ) );
}

void BatchWizard::setupPage2()
{
    page2 = new TQWidget( this, "page2" );
    TQVBoxLayout* pageLayout = new TQVBoxLayout( page2, 11, 6, "pageLayout");

    TQVButtonGroup* group = new TQVButtonGroup( page2 );
    
    radioSimple = new TQRadioButton( i18n("Print &labels without data"), group );
    radioSqlArticles = new TQRadioButton( i18n("Print &articles from KBarcodes SQL database"), group );
    radioVarImport = new TQRadioButton( i18n("Import &variables and print"), group );
    radioAddressBook = new TQRadioButton( i18n("Print &contacts from your addressbook"), group );
    radioSimple->setChecked( true );

    TQSpacerItem* spacer = new TQSpacerItem( 20, 20, TQSizePolicy::Expanding, TQSizePolicy::Expanding );
    pageLayout->addWidget( group );
    pageLayout->addItem( spacer );

    connect( radioSimple, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );
    connect( radioSqlArticles, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );
    connect( radioVarImport, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );
    connect( radioAddressBook, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );

    addPage( page2, i18n("Data Source") );
}

void BatchWizard::setupPage3()
{
    page3 = new TQWidgetStack( this, "page3" );

    setupStackPage1();
    setupStackPage2();
    setupStackPage3();
    setupStackPage4();

    addPage( page3, i18n("Print Data") );
}

void BatchWizard::setupPage4()
{
    page4 = new TQVBox( this, "page4" );
    page4->setSpacing( 5 );

    TQHBox* hbox = new TQHBox( page4 );
    hbox->setSpacing( 5 );
    
    buttonTableInsert = new KPushButton( i18n("Insert Row"), hbox );
    buttonTableInsert->setIconSet( BarIconSet( "edit" ) );
    buttonTableRemove = new KPushButton( i18n("Delete Row"), hbox );
    buttonTableRemove->setIconSet( BarIconSet( "edit-delete") );

    m_varTable = new TQTable( page4 );
    m_varTable->setReadOnly( false );
    m_varTable->setSelectionMode( TQTable::SingleRow );

    addPage( page4, i18n("Import Variables") );

    connect( buttonTableInsert, TQT_SIGNAL( clicked() ), this, TQT_SLOT( slotTableInsert() ) );
    connect( buttonTableRemove, TQT_SIGNAL( clicked() ), this, TQT_SLOT( slotTableRemove() ) );
}

void BatchWizard::setupPage5()
{
    TokenProvider serial( TQT_TQPAINTDEVICE(this) );

    page5 = new TQVBox( this, "page5" );

    new TQLabel( i18n( "<qt>KBarcode has support for placing serial numbers on labels. "
		      "If you did not use the [serial] token on your label in "
		      "a text field or a barcode, you can skip this page.<br>"
		      "Serial start is a free form start value containing at least one "
		      "number. This number is increased for every printed label on the "
		      "print out.</qt>"), page5 );

    TQHBox* hbox = new TQHBox( page5 );
    hbox->setSpacing( 5 );
    
    new TQLabel( i18n( "Serial start:" ), hbox );
    serialStart = new KLineEdit( serial.serial(), hbox );

    serialInc = new KIntNumInput( 1, hbox );
    serialInc->setLabel( i18n( "Serial increment:" ), KNumInput::AlignLeft | KNumInput::AlignVCenter );
    serialInc->setRange( 1, 10000, 1, false );

    addPage( page5, i18n("Serial Number") );
}

void BatchWizard::setupPage10()
{
    page10 = new TQWidget( this, "page10" );
    TQVBoxLayout* pageLayout = new TQVBoxLayout( page10, 11, 6, "pageLayout");

    TQVButtonGroup* group = new TQVButtonGroup( page10 );
    
    radioPrinter = new TQRadioButton( i18n("&Print to a system printer or to a file"), group );
    radioImage = new TQRadioButton( i18n("&Create images"), group );

    imageBox = new TQVBox( group );
    imageBox->setMargin( 10 );

    radioBarcode = new TQRadioButton( i18n("Print to a special &barcode printer"), group );

    TQHBox* directoryBox = new TQHBox( imageBox );
    directoryBox->setSpacing( 5 );
    TQLabel* label = new TQLabel( i18n("Output &Directory:"), directoryBox );
    imageDirPath = new KURLRequester( directoryBox );
    imageDirPath->setMode( KFile::Directory | KFile::ExistingOnly | KFile::LocalOnly );
    label->setBuddy( directoryBox );

    TQHBox* formatBox = new TQHBox( imageBox );
    label = new TQLabel( i18n("Output File &Format:"), formatBox );

    TQStringList formats = KImageIO::types( KImageIO::Writing );
    comboFormat = new KComboBox( false, formatBox );
    comboFormat->insertStringList( formats );
    if( formats.contains( PNG_FORMAT ) )
	comboFormat->setCurrentItem( formats.findIndex( PNG_FORMAT ) );
    label->setBuddy( comboFormat );

    TQVButtonGroup* imageNameGroup = new  TQVButtonGroup( i18n("&Filename:"), imageBox );
    radioImageFilenameArticle = new TQRadioButton( i18n("Use &article number for filename"), imageNameGroup );
    radioImageFilenameBarcode = new TQRadioButton( i18n("Use &barcode number for filename"), imageNameGroup );
    radioImageFilenameCustom  = new TQRadioButton( i18n("Use &custom filename:"), imageNameGroup );
    editImageFilename = new KLineEdit( imageNameGroup );
    radioImageFilenameBarcode->setChecked( true );

    labelInfo = new TQLabel( page10 );

    radioPrinter->setChecked( true );

    checkKeepOpen = new TQCheckBox( i18n("&Keep window open after printing."), page10 );

    TQSpacerItem* spacer = new TQSpacerItem( 20, 20, TQSizePolicy::Expanding, TQSizePolicy::Expanding );
    pageLayout->addWidget( group );
    pageLayout->addItem( spacer );
    pageLayout->addWidget( labelInfo );
    pageLayout->addWidget( checkKeepOpen );

    connect( radioPrinter, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );
    connect( radioImage, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );
    connect( radioBarcode, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );

    connect( radioImageFilenameArticle, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );
    connect( radioImageFilenameBarcode, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );
    connect( radioImageFilenameCustom, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );

    connect( imageDirPath, TQT_SIGNAL( textChanged( const TQString & ) ), this, TQT_SLOT( enableControls() ) );

    addPage( page10, i18n("Output Device") );
}

void BatchWizard::setupStackPage1()
{
    stack1 = new TQVBox( page3, "stack1" );
    stack1->setSpacing( 5 );

    TQHBox* hbox = new TQHBox( stack1 );
    hbox->setSpacing( 5 );

    new TQLabel( i18n( "Customer name and no.:" ), hbox );
    customerName = new KComboBox( false, hbox );
    customerId = new KComboBox( false, hbox );

    TQHBox* hButtonBox = new TQHBox( stack1 );
    hButtonBox->setSpacing( 5 );

    buttonAdd = new KPushButton( i18n( "&Add..." ), hButtonBox );
    buttonImport = new KPushButton( i18n("&Import..."), hButtonBox );
    buttonEdit = new KPushButton( i18n( "&Edit..." ), hButtonBox );
    buttonRemove = new KPushButton( i18n("&Remove" ), hButtonBox );
    buttonRemoveAll = new KPushButton( i18n("R&emove All"), hButtonBox );

    TDEPopupMenu* mnuImport = new TDEPopupMenu( this );
    mnuImport->insertItem( i18n("Import from File ..."), this, TQT_SLOT( loadFromFile() ) );
    mnuImport->insertItem( i18n("Import from Clipboard ..."), this, TQT_SLOT( loadFromClipboard() ) );
    mnuImport->insertItem( i18n("Import barcode_basic"), this, TQT_SLOT( addAllItems() ) );
    buttonImport->setPopup( mnuImport );

    sqlList = new TDEListView( stack1 );
    sqlList->addColumn( i18n("Index") );
    sqlList->addColumn( i18n("Number of Labels") );
    sqlList->addColumn( i18n("Article Number") );
    sqlList->addColumn( i18n("Group") );
    sqlList->setAllColumnsShowFocus( true );
    connect( sqlList, TQT_SIGNAL(doubleClicked(TQListViewItem*,const TQPoint &,int)),
             this, TQT_SLOT(changeItem(TQListViewItem*,const TQPoint &,int)));

    connect( customerName, TQT_SIGNAL( activated(int) ), this, TQT_SLOT( customerNameChanged(int) ) );
    connect( customerId, TQT_SIGNAL( activated(int) ), this, TQT_SLOT( customerIdChanged(int) ) );
    connect( buttonAdd, TQT_SIGNAL( clicked() ), this, TQT_SLOT( addItem() ) );
    connect( buttonEdit, TQT_SIGNAL( clicked() ), this, TQT_SLOT( editItem() ) );
    connect( buttonRemove, TQT_SIGNAL( clicked() ), this, TQT_SLOT( removeItem() ) );
    connect( buttonRemoveAll, TQT_SIGNAL( clicked() ), sqlList, TQT_SLOT( clear() ) );
    connect( buttonRemoveAll, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );

    page3->addWidget( stack1 );
}

void BatchWizard::setupStackPage2()
{
    stack2 = new TQHBox( page3, "stack2" );
    stack2->setSpacing( 5 );

    TQVButtonGroup* group = new TQVButtonGroup( stack2 );
    radioImportManual = new TQRadioButton( i18n("Enter &data manually"), group );
    radioImportSql = new TQRadioButton( i18n("Import variables from a &SQL table"), group );
    labelSqlQuery = new TQLabel( i18n("Please enter a sql &query:"), group );
    importSqlQuery = new KLineEdit( group );
    labelSqlQuery->setBuddy( importSqlQuery );

    radioImportCSV = new TQRadioButton( i18n("Import from a &CSV file"), group );
    labelCsvFile= new TQLabel( i18n("Please select a csv &file:"), group );
    importCsvFile = new KURLRequester( group );
    labelCsvFile->setBuddy( importCsvFile );
    labelEncoding = new TQLabel( i18n("&Encoding:"), group );
    comboEncoding = new EncodingCombo( group );
    labelEncoding->setBuddy( comboEncoding );

    radioImportManual->setChecked( true );

    TQVBox* box = new TQVBox( stack2 );
    box->setSpacing( 5 );

    new TQLabel( i18n("Available Variables:"), box );
    m_varList = new TDEListBox( box );

    connect( radioImportManual, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );
    connect( radioImportSql, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );
    connect( radioImportCSV, TQT_SIGNAL( clicked() ), this, TQT_SLOT( enableControls() ) );
    connect( importSqlQuery, TQT_SIGNAL( textChanged( const TQString & ) ), this, TQT_SLOT( enableControls() ) );
    connect( importCsvFile, TQT_SIGNAL( textChanged( const TQString & ) ), this, TQT_SLOT( enableControls() ) );

    page3->addWidget( stack2 );
}

void BatchWizard::setupStackPage3()
{
    stack3 = new TQVBox( page3, "stack3" );

    numLabels = new KIntNumInput( 1, stack3 );
    numLabels->setRange( 1, 100000, 1, true );
    numLabels->setLabel( i18n("&Number of labels to print:"), AlignLeft | AlignVCenter );

    page3->addWidget( stack3 );
}

void BatchWizard::setupStackPage4()
{
    stack4 = new TQWidget( page3, "stack4" );
    
    TQHBoxLayout* mainLayout = new TQHBoxLayout( stack4 );

    TQVBox* list1 = new TQVBox( stack4 );
    TQVBox* list2 = new TQVBox( stack4 );

    TQFrame* buttons = new TQFrame( stack4 );
    buttons->setMargin( 10 );

    TQVBoxLayout* layout = new TQVBoxLayout( buttons );
    TQSpacerItem* spacer1 = new TQSpacerItem( 20, 20, TQSizePolicy::Expanding, TQSizePolicy::Expanding );
    TQSpacerItem* spacer2 = new TQSpacerItem( 20, 20, TQSizePolicy::Expanding, TQSizePolicy::Expanding );

    buttonAddAllAddress  = new KPushButton( buttons );
    buttonAddAddress = new KPushButton( buttons );
    buttonRemoveAddress = new KPushButton( buttons );;
    buttonRemoveAllAddress = new KPushButton( buttons );

    buttonAddAllAddress->setIconSet( BarIconSet( "2rightarrow" ) );
    buttonAddAddress->setIconSet( BarIconSet( "1rightarrow" ) );
    buttonRemoveAddress->setIconSet( BarIconSet( "1leftarrow" ) );
    buttonRemoveAllAddress->setIconSet( BarIconSet( "2leftarrow" ) );

    TQToolTip::add( buttonAddAllAddress, i18n("Add all contacts to the list of contacts which will be printed.") );
    TQToolTip::add( buttonAddAddress, i18n("Add selected contacts to the list of contacts which will be printed.") );
    TQToolTip::add( buttonRemoveAddress, i18n("Remove selected contacts from the list of contacts which will be printed.") );
    TQToolTip::add( buttonRemoveAllAddress, i18n("Remove all contacts from the list of contacts which will be printed.") );

    layout->addItem( spacer1 );
    layout->addWidget( buttonAddAllAddress );
    layout->addWidget( buttonAddAddress );
    layout->addWidget( buttonRemoveAddress );
    layout->addWidget( buttonRemoveAllAddress );
    layout->addItem( spacer2 );

    mainLayout->addWidget( list1 );
    mainLayout->addWidget( buttons );
    mainLayout->addWidget( list2 );

    mainLayout->setStretchFactor( list1, 2 );
    mainLayout->setStretchFactor( list2, 2 );

    new TQLabel( i18n("All Addresses"), list1 );
    new TQLabel( i18n("Selected Addresses"), list2 );

    listAddress = new TDEListView( list1 );
    listAddress->addColumn( i18n("Given Name"), 0 );
    listAddress->addColumn( i18n("Family Name"), 1 );
    listAddress->addColumn( i18n("Email Address"), 2 );
    listAddress->setMultiSelection( true );
    listAddress->setAllColumnsShowFocus( true );

    listAddress->setColumnWidthMode( 0, TQListView::Maximum );
    listAddress->setColumnWidthMode( 1, TQListView::Maximum );
    listAddress->setColumnWidthMode( 2, TQListView::Maximum );

    listSelectedAddress = new TDEListView( list2 );
    listSelectedAddress->addColumn( i18n("Given Name"), 0 );
    listSelectedAddress->addColumn( i18n("Family Name"), 1 );
    listSelectedAddress->addColumn( i18n("Email Address"), 2 );
    listSelectedAddress->setMultiSelection( true );
    listSelectedAddress->setAllColumnsShowFocus( true );

    listSelectedAddress->setColumnWidthMode( 0, TQListView::Maximum );
    listSelectedAddress->setColumnWidthMode( 1, TQListView::Maximum );
    listSelectedAddress->setColumnWidthMode( 2, TQListView::Maximum );

    connect( buttonAddAddress, TQT_SIGNAL( clicked() ), this, TQT_SLOT( slotAddAddress() ) );
    connect( buttonRemoveAddress, TQT_SIGNAL( clicked() ), this, TQT_SLOT( slotRemoveAddress() ) );
    connect( buttonAddAllAddress, TQT_SIGNAL( clicked() ), this, TQT_SLOT( slotAddAllAddress() ) );
    connect( buttonRemoveAllAddress, TQT_SIGNAL( clicked() ), this, TQT_SLOT( slotRemoveAllAddress() ) );

    page3->addWidget( stack4 );
}

void BatchWizard::setupSql()
{
    SqlTables* tables = SqlTables::getInstance();
    if( !tables->isConnected() )
        return;

    TQSqlCursor cur( TABLE_CUSTOMER );
    cur.select();
    customerId->clear();
    customerName->clear();
    while ( cur.next() ) {
        customerId->insertItem( cur.value("customer_no" ).toString() );
        customerName->insertItem( cur.value("customer_name" ).toString() );
    }
}

void BatchWizard::enableControls()
{
    setAppropriate( page4, radioVarImport->isChecked() );

    radioSqlArticles->setEnabled( SqlTables::getInstance()->isConnected() );
    radioImportSql->setEnabled( SqlTables::getInstance()->isConnected() );

    importCsvFile->setEnabled( radioImportCSV->isChecked() );
    labelCsvFile->setEnabled( radioImportCSV->isChecked() );
    importSqlQuery->setEnabled( radioImportSql->isChecked() );
    labelSqlQuery->setEnabled( radioImportSql->isChecked() );
    labelEncoding->setEnabled( radioImportCSV->isChecked() );
    comboEncoding->setEnabled( radioImportCSV->isChecked() );

    buttonRemove->setEnabled( sqlList->childCount() );
    buttonRemoveAll->setEnabled(sqlList->childCount() );
    buttonEdit->setEnabled( sqlList->childCount() );

    imageBox->setEnabled( radioImage->isChecked() );

    if( radioImportSql->isChecked() )
	setNextEnabled( page3, !importSqlQuery->text().isEmpty() );
    else if( radioImportCSV->isChecked() )
	setNextEnabled( page3, !importCsvFile->url().isEmpty() );
    else if( radioImportManual->isChecked() )
	setNextEnabled( page3, true );

    editImageFilename->setEnabled( radioImageFilenameCustom->isChecked() );
    radioImageFilenameArticle->setEnabled( radioSqlArticles->isChecked() );

    setNextEnabled( page1, !m_url->url().isEmpty() );
    
    if( radioAddressBook->isChecked() )
        setNextEnabled( page3, listSelectedAddress->childCount() );

    if( radioImage->isChecked() )
	setFinishEnabled( page10, !imageDirPath->url().isEmpty() );
    else
	setFinishEnabled( page10, true );
}

void BatchWizard::showPage( TQWidget* p )
{
    if( p == page3 )
    {
	if( radioSqlArticles->isChecked() )
	    page3->raiseWidget( stack1 );
	else if( radioVarImport->isChecked() )
	{
	    page3->raiseWidget( stack2 );
	    fillVarList();
	}
	else if( radioSimple->isChecked() )
	    page3->raiseWidget( stack3 );
        else if( radioAddressBook->isChecked() )
        {
            page3->raiseWidget( stack4 );
            fillAddressList();
        }
    }
    else if( p == page4 )
	if( !fillVarTable() )
	    return;


    KWizard::showPage( p );
}

void BatchWizard::accept()
{
    printNow( TQString() );
}

void BatchWizard::printNow( const TQString & printer, bool bUserInteraction )
{
    BatchPrinter* batch = NULL;
    KPrinter* prn = NULL;
    int batchType = 0;

    // let's check if the label file does even exist!
    if( !TQFile::exists( m_url->url() ) ) 
    {
        KMessageBox::error( this, TQString( i18n("The label file %1 was not found") ).arg( m_url->url()) );
        return;
    }

    if( radioPrinter->isChecked() )
    {
	int move = 0;
        if( bUserInteraction ) 
        {
            PrintLabelDlg pld( this, "pld" );
            pld.setLabelsEnabled( false );
            if( pld.exec() != TQDialog::Accepted )
		return;
	    
            move = pld.position();
            PrinterSettings::getInstance()->getData()->border = pld.border();
        }
	
	prn = PrinterSettings::getInstance()->setupPrinter( m_url->url(), this, !printer.isEmpty(), printer );
	if( !prn ) 
	    return;

	batch = new BatchPrinter( prn, this );
	batch->setMove( move );
     
	batchType = BatchPrinter::POSTSCRIPT;
    }
    else if( radioBarcode->isChecked() )
    {
	BarcodePrinterDlg dlg(this);
	if( dlg.exec() != TQDialog::Accepted )
	    return;

	batch = new BatchPrinter( dlg.printToFile() ? dlg.fileName() : dlg.deviceName(), 
				  dlg.outputFormat(), this );
	batchType = BatchPrinter::BCP;
    }
    else if( radioImage->isChecked() )
    {
	batch = new BatchPrinter( imageDirPath->url(), this );
	if( radioImageFilenameArticle->isChecked() )
	    batch->setImageFilename( BatchPrinter::E_ARTICLE );
	else if( radioImageFilenameBarcode->isChecked() )
	    batch->setImageFilename( BatchPrinter::E_BARCODE );
	else if( radioImageFilenameCustom->isChecked() )
	{
	    batch->setImageFilename( BatchPrinter::E_CUSTOM );
	    batch->setImageCustomFilename( editImageFilename->text() );
	}

	batchType =  BatchPrinter::IMAGE;
    }

    if( !checkKeepOpen->isChecked() )
        KWizard::accept();

    TDEApplication::setOverrideCursor( TQCursor( TQt::ArrowCursor ), true );
    setupBatchPrinter( batch, batchType );
    TDEApplication::restoreOverrideCursor();

    delete prn;
    delete batch;
}

void BatchWizard::setupBatchPrinter( BatchPrinter* batch, int m )
{
    Definition* def = NULL;
    TQString description;
    bool kbarcode18;

    fillByteArray();
    TQDomDocument doc( "KBarcodeLabel" );
    if ( !doc.setContent( m_bytearray ) ) 
        return;
    
    XMLUtils util;
    util.readXMLHeader( &doc, description, kbarcode18, &def );

    TQBuffer buffer( m_bytearray );
    if( !buffer.open( IO_ReadOnly ) )
        return;

    batch->setBuffer( &buffer );
    batch->setSerial( serialStart->text(), serialInc->value() );
    batch->setName( m_url->url() );
    batch->setDefinition( def );
    batch->setImageFormat( comboFormat->currentText() );

    if( radioSqlArticles->isChecked() )
    {
	int labels = 0;
	batch->setCustomer( customerId->currentText() );
	
        // sort by group
	sqlList->setSorting( 3, true );
	sqlList->sort();

	TQValueList<BatchPrinter::data>* dlist = new TQValueList<BatchPrinter::data>;
	TQListViewItem* item = sqlList->firstChild();
	while( item ) 
	{
	    BatchPrinter::data m_data;
	    m_data.number = item->text( 1 ).toInt();
	    labels += m_data.number;
	    m_data.article_no = item->text( 2 );
	    m_data.group = item->text( 3 );
	    
	    dlist->append( m_data );
	    item = item->nextSibling();
	};

	batch->setData( dlist );
	batch->setLabels( labels );
    }
    else if( radioSimple->isChecked() )
    {
	batch->setLabels( numLabels->value() );

	// do a dirty drick, TODO: refactor BatchPrinter in the future
	TQValueList<BatchPrinter::data>* dlist = new TQValueList<BatchPrinter::data>;
	BatchPrinter::data m_data;
	m_data.number = numLabels->value();
	dlist->append( m_data );

	batch->setData( dlist );	
    }
    else if( radioVarImport->isChecked() )
    {
	TVariableList* tVariableList = new TVariableList;
	for( int i=0; i<m_varTable->numRows(); i++ )
	{
	    TQMap<TQString, TQString> map;
	    for( int z=0; z<m_varTable->numCols(); z++ )
		map[ m_varTable->horizontalHeader()->label( z ) ] = m_varTable->text( i, z );
	    tVariableList->append( map );
	}

	batch->setData( tVariableList );
    }
    else if( radioAddressBook->isChecked() )
    {
        TDEABC::AddresseeList* list = new TDEABC::AddresseeList;
        TQListViewItem* item = listSelectedAddress->firstChild();
        while( item ) 
        {
            list->append( static_cast<AddressListViewItem*>(item)->address() );
            item = item->nextSibling();
        }

        batch->setData( list );
    }

    if( m == BatchPrinter::POSTSCRIPT )    
        batch->start();
    else if( m == BatchPrinter::IMAGE )
        batch->startImages();
    else if( m == BatchPrinter::BCP )
        batch->startBCP();

    delete def;
}


void BatchWizard::addItem()
{
    DSSmallDialogs::AddItemsDialog aid( this, "aid" );
    aid.setGroupCompletion( compGroup );
    connect( &aid, TQT_SIGNAL( add( const TQString &, const TQString &, int) ),
             this, TQT_SLOT( slotAddItem( const TQString &, const TQString &, int) ) );

    aid.exec();
}

bool BatchWizard::slotAddItem( const TQString & article, const TQString & group, int count )
{
    return this->addItem( article, group, count, true );
}

bool BatchWizard::addItem( const TQString & article, const TQString & group, int count, bool msg )
{
    if( !article.isEmpty() && !existsArticle( article ) ) {
        if( msg )
            KMessageBox::error( this, i18n("Please enter a valid article ID") );
        return false;
    }

    TQString temp;
    temp.sprintf("%0*i", 5, sqlList->childCount() + 1 );

    TDEListViewItem* item = new TDEListViewItem( sqlList, temp, TQString( "%1" ).arg( count ),
                          article, group );
    sqlList->insertItem( item );

    addGroupCompletion( group );
    enableControls();

    return true;
}

void BatchWizard::addGroupCompletion( const TQString & group )
{
    if( !group.isEmpty() ) 
    {
        TQStringList slist = compGroup->items();
        if(!slist.contains( group ) )
            compGroup->addItem( group );
    }
}

bool BatchWizard::existsArticle( const TQString & article )
{
    if( article.isEmpty() )
        return false;

    TQSqlQuery query( "select uid from barcode_basic where article_no='" + article + "'" );
    while ( query.next() )
        return true;

    return false;
}

void BatchWizard::editItem()
{
    TQListViewItem* item = sqlList->selectedItem();
    if( item )
        changeItem( item, TQPoint(0,0), 0 );
}

void BatchWizard::changeItem( TQListViewItem* item, const TQPoint &, int )
{
    if(!item)
        return;

    DSSmallDialogs::AddItemsDialog aid( item->text( 2 ), item->text( 3 ),
                                        item->text( 1 ).toInt(), this, "aid" );
    aid.setGroupCompletion( compGroup );

    if( aid.exec() == TQDialog::Accepted ) 
    {
        item->setText( 1, TQString::number( aid.count() ) );
        item->setText( 2, aid.articleNo() );
        item->setText( 3, aid.groupName() );
        addGroupCompletion( aid.groupName() );
	enableControls();
    }
}

void BatchWizard::removeItem() 
{
    TQListViewItem* item = sqlList->firstChild();
    while( item ) 
    {
        if( item->isSelected() ) 
	{
            TQListViewItem* it = item->nextSibling();
            delete item;

            while( it ) 
	    {
                int a = it->text( 0 ).toInt();
                TQString temp;
                temp.sprintf("%0*i", 5, a - 1 );
                it->setText( 0, temp );
                it = it->nextSibling();
            }

            break;
        } else
            item = item->nextSibling();
    }

    enableControls();
}

void BatchWizard::customerIdChanged( int index ) 
{
    customerName->setCurrentItem( index );
    enableControls();
}

void BatchWizard::customerNameChanged( int index ) 
{
    customerId->setCurrentItem( index );
    enableControls();
}

void BatchWizard::addAllItems() 
{
    DSSmallDialogs::AddAllDialog* dlg = new DSSmallDialogs::AddAllDialog( this, "dlg" );
    if( dlg->exec() == TQDialog::Accepted )
    {
	TQString temp;
	TQString group = dlg->groupName();
	const TQString num = TQString::number( dlg->numberLabels() );

	TQSqlQuery query("SELECT article_no FROM " TABLE_BASIC );
	while( query.next() ) 
	{
	    temp.sprintf("%0*i", 5, sqlList->childCount() + 1 );
	    new TDEListViewItem( sqlList, temp, num, query.value( 0 ).toString(), group );
	}

        enableControls();
    }
}

void BatchWizard::loadFromFile() 
{
    TQString f = KFileDialog::getOpenFileName( 0, 0, this );
    if( !f.isEmpty() )
        loadFromFile( f );
}

void BatchWizard::loadFromClipboard() 
{
    TQClipboard *cb = TDEApplication::clipboard();
    loadData( cb->text() );
}

void BatchWizard::loadFromFile( const TQString & url ) 
{
    TQByteArray data;
    TQFile file( url );
    
    if( !file.open( IO_ReadOnly ) ) 
    {
        tqDebug("Unable to open file: %s", url.latin1() );
        return;
    }

    data = file.readAll();

    loadData( TQString( data ) );
}

void BatchWizard::loadData( const TQString & data ) 
{
    labelprinterdata* lpdata = PrinterSettings::getInstance()->getData();
    if( lpdata->separator.isEmpty() ) 
    {
        KMessageBox::sorry( this, i18n("Separator is empty. Please set it to a value.") );
        return;
    }

    // new config entry!!!
    TDEConfig* config = kapp->config();
    config->setGroup("FileFormat");
    int pos[3] = { config->readNumEntry("Data0", 0 ),
                   config->readNumEntry("Data1", 1 ),
                   config->readNumEntry("Data2", 2 ) };

    bool custom_article_no = lpdata->useCustomNo;
    TQBuffer buf( data.utf8() );
    CSVFile file( buf );

    TQStringList list, dropped;
    TQString article, quantity, group;

    while( file.isValid() && !file.isEof() )
    {
	list = file.readNextLine();
	while( list.count() < 3 )
	    list.append( TQString() );

	if( pos[0] == 0 )
	    quantity = list[0];
	else if( pos[0] == 1 )
	    article = list[0];
	else
	    group = list[0];

	if( pos[1] == 0 )
	    quantity = list[1];
	else if( pos[1] == 1 )
	    article = list[1];
	else
	    group = list[1];

	if( pos[2] == 0 )
	    quantity = list[2];
	else if( pos[2] == 1 )
	    article = list[2];
	else
	    group = list[2];

	// data[0] == quantity
	// data[1] == article_no
	// data[2] == group

	bool qint = false;
	(void)quantity.toInt( &qint );

	if( qint && custom_article_no ) {
	    qint = false;
	    TQSqlQuery query("SELECT article_no FROM customer_text WHERE article_no_customer='" + article + "'" );
	    while( query.next() ) {
		article = query.value( 0 ).toString();
		qint = true;
		break;
	    }
	}

	if( qint ) // && existsArticle( article )
	{    
	    if( !addItem( TQString( article ), TQString( group ), quantity.toInt(), false ) )
		dropped.append( quantity + lpdata->separator +  article + lpdata->separator + group );
	}
    }

    if( !dropped.isEmpty() )
        KMessageBox::informationList( this, i18n("<qt>The following items can not be added:" )+ "</qt>", dropped );

    enableControls();
}

void BatchWizard::fillByteArray()
{
    if( m_bytearray_filename != m_url->url() )
    {
	TQFile f( m_url->url() );
	if ( !f.open( IO_ReadOnly ) )
	{
	    m_bytearray_filename = TQString();
	    m_bytearray.resize( 0 );
	    return ;
	}

	m_bytearray = f.readAll();
	f.close();
    }
}

void BatchWizard::fillVarList()
{
    fillByteArray();
    TQDomDocument doc( "KBarcodeLabel" );
    if ( !doc.setContent( m_bytearray ) ) 
        return;
    
    XMLUtils util;
    DocumentItemList list;
    list.setAutoDelete( true );

    TokenProvider token( TQT_TQPAINTDEVICE(this) );
    Definition* def = NULL;

    TQString description;
    bool kbarcode18;
    util.readXMLHeader( &doc, description, kbarcode18, &def );
    util.readDocumentItems( &list, &doc, &token, kbarcode18 );

    token.setCurrentDocumentItems( list );

    TQStringList vars = token.listUserVars();
    m_varList->clear();
    m_varList->insertStringList( vars );
    m_varTable->setNumCols( vars.count() );
    for( unsigned int i = 0; i < vars.count(); i++ )
    {
	vars[i] = vars[i].right( vars[i].length() - 1 );
	m_varTable->horizontalHeader()->setLabel( i, vars[i] );
    }

    delete def;
}

void BatchWizard::fillAddressList()
{
    TDEABC::AddressBook* ab = TDEABC::StdAddressBook::self();
    listAddress->clear();
    
    TDEABC::AddressBook::Iterator it;
    listAddress->setUpdatesEnabled( false );
    for ( it = ab->begin(); it != ab->end(); ++it ) 
        new AddressListViewItem( listAddress, *it );
    listAddress->setUpdatesEnabled( true );
}

bool BatchWizard::fillVarTable()
{
    // Clear the table
    m_varTable->setNumRows( 0 );

    if( radioImportSql->isChecked() )
    {
	int y = 0;
	int x;
	TQSqlSelectCursor query( importSqlQuery->text(), SqlTables::getInstance()->database() );
	query.select();
	if( query.lastError().type() != TQSqlError::None )
	{
	    KMessageBox::error( this, i18n("<qt>Can't execute SQL query:<br>") + query.lastError().text() + "</qt>" );
	    return false;
	}

	if( m_varTable->numRows() < query.size() )
	    m_varTable->setNumRows( query.size() );

	while( query.next() )
	{
	    for( x=0;x<m_varTable->numRows();x++ )
		m_varTable->setText( y, x, query.value( m_varTable->horizontalHeader()->label( x ) ).toString() );

	    y++;
	}
    }
    else if( radioImportCSV->isChecked() )
    {
	CSVFile file( importCsvFile->url() );
        file.setEncoding( comboEncoding->currentText() );

	TQStringList heading;
	TQStringList data;
	int i = 0;

        file.setCSVFile(true);
	if( !file.isValid() )
	{
	    KMessageBox::error( this, TQString( i18n("Can't open file: %1") ).arg( importCsvFile->url() ) );
	    return false;
	}

	while( !file.isEof() )
	{
	    if( heading.isEmpty() )
		heading = file.readNextLine();
            else
	    {
		data = file.readNextLine();

		// add 100 rows to get a reasonable speed
		if( m_varTable->numRows() <= i )
		    m_varTable->setNumRows( i + 100 );	      

                printf("datacount=%i\n", data.count() );
		for( unsigned int z = 0; z < data.count(); z++ )
		{
                    printf("numRows=%i\n", m_varTable->numCols() );
		    for( int x = 0; x < m_varTable->numCols(); x++ )
                    {
                        printf("horizontal header=%s\n", m_varTable->horizontalHeader()->label( x ).lower().latin1() );
                        printf("heading=%s\n", heading[z].lower().latin1() );
			if( m_varTable->horizontalHeader()->label( x ).lower() == heading[z].lower() )
			{
                            printf("Reading: (%s)\n", data[z].latin1());
			    m_varTable->setText( i, x, data[z] );
			    break;
			}
                    }
		}

		if( data.count() )
		    i++;
            }
	}

	m_varTable->setNumRows( i );
    }

    return true;
}

void BatchWizard::slotTableInsert()
{
    m_varTable->insertRows( m_varTable->numRows(), 1 );
}

void BatchWizard::slotTableRemove()
{
    TQTableSelection sel = m_varTable->selection( m_varTable->currentSelection() );
    m_varTable->removeRow( sel.topRow() );
}

void BatchWizard::setFilename( const TQString & url )
{
    m_url->setURL( url );
    enableControls();
}

void BatchWizard::setImportSqlQuery( const TQString & query )
{
    radioImportCSV->setChecked( false );
    radioImportManual->setChecked( false );
    radioImportSql->setChecked( true );

    radioVarImport->setChecked( true );
    radioSqlArticles->setChecked( false );
    radioSimple->setChecked( false );

    importSqlQuery->setText( query );

    enableControls();

    showPage( page3 );
    showPage( page4 );
}

void BatchWizard::setImportCsvFile( const TQString & filename )
{
    radioImportCSV->setChecked( true );
    radioImportManual->setChecked( false );
    radioImportSql->setChecked( false );

    radioVarImport->setChecked( true );
    radioSqlArticles->setChecked( false );
    radioSimple->setChecked( false );

    importCsvFile->setURL( filename );

    enableControls();

    showPage( page3 );
    showPage( page4 );
}

void BatchWizard::setNumLabels( const int n )
{
    numLabels->setValue( n );
    radioSimple->setChecked( true );
    radioSqlArticles->setChecked( false );
    radioVarImport->setChecked( false );
    enableControls();
}

void BatchWizard::setOutputFormat( const int e )
{
    radioBarcode->setChecked( false );
    radioImage->setChecked( false );
    radioPrinter->setChecked( false );

    switch( e )
    {
	case BatchPrinter::BCP:
	    radioBarcode->setChecked( true );
	    break;
	case BatchPrinter::IMAGE:
	    radioImage->setChecked( true );
	    break;
	default:
	case BatchPrinter::POSTSCRIPT:
	    radioPrinter->setChecked( true );
	    break;
    }

    enableControls();
}

void BatchWizard::setSerialNumber( const TQString & val, int inc )
{
    serialInc->setValue( inc );
    serialStart->setText( val );

    // Not needed here: enableControls();
}

void BatchWizard::slotAddAddress()
{
    moveAddress( listAddress, listSelectedAddress );
    enableControls();
}

void BatchWizard::slotAddAllAddress()
{
    moveAddress( listAddress, listSelectedAddress, true );
    enableControls();
}

void BatchWizard::slotRemoveAddress()
{
    moveAddress( listSelectedAddress, listAddress );
    enableControls();
}

void BatchWizard::slotRemoveAllAddress()
{
    moveAddress( listSelectedAddress, listAddress, true );
    enableControls();
}

void BatchWizard::moveAddress( TQListView* src, TQListView* dst, bool bAll )
{
    TQListViewItem* item = src->firstChild();
    TQListViewItem* cur;

    while( item ) 
    {
        if( bAll || item->isSelected() )
        {
            cur = item;
            item = item->nextSibling();

            src->takeItem( cur );
            dst->insertItem( cur );
            cur->setSelected( false );
        }
        else
            item = item->nextSibling();
    }
}
