/*
Copyright (c) 2003 Jos van den Oever

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*/
// TQt
#include <tqlabel.h>
#include <tqtextedit.h>
#include <tqfileinfo.h>

// KDE
#include <tdeversion.h>
#include <tdelocale.h>

// Local
#include "gvcore/document.h"
#include "metaedit.moc"
namespace Gwenview {

// FIXME: Why doesn't MetaEdit inherits from TQTextEdit rather than TQVBox?
MetaEdit::MetaEdit(TQWidget *parent, Document *gvp, const char *name)
: TQVBox(parent, name)
, mEmpty(true)
, mDocument(gvp)
{
	mCommentEdit=new TQTextEdit(this);
	mCommentEdit->installEventFilter(this);
	connect(mCommentEdit, TQT_SIGNAL(modificationChanged(bool)),
		this, TQT_SLOT(setModified(bool)));
	connect(mDocument,TQT_SIGNAL(loaded(const KURL&)),
		this,TQT_SLOT(updateContent()) );
	connect(mCommentEdit, TQT_SIGNAL(textChanged()),
		this, TQT_SLOT(updateDoc()) );
	updateContent();
	mCommentEdit->setMinimumHeight(int (mCommentEdit->fontMetrics().height() * 1.5) );
}


MetaEdit::~MetaEdit() {
}


bool MetaEdit::eventFilter(TQObject*, TQEvent *event) {
	if (mEmpty
		&& (mDocument->commentState()==Document::WRITABLE)
		&& (event->type()==TQEvent::FocusIn || event->type()==TQEvent::FocusOut) 
	) {
		setEmptyText();
	}
	return false;
}


void MetaEdit::setModified(bool m) {
	if (m && mEmpty) {
		mEmpty = false;
	}
}


void MetaEdit::updateContent() {
	if (mDocument->isNull()) {
		setMessage(i18n("No image selected."));
		return;
	}

	if (mDocument->commentState() == Document::NONE) {
		setMessage(i18n("This image cannot be commented."));
		return;
	}
	
	TQString comment=mDocument->comment();
	mEmpty = comment.isEmpty();
	if (mEmpty) {
		setEmptyText();
		return;
	}
	setComment(comment);
}


void MetaEdit::updateDoc() {
	if ((mDocument->commentState()==Document::WRITABLE) && mCommentEdit->isModified()) {
		mDocument->setComment(mCommentEdit->text());
		mCommentEdit->setModified(false);
	}
}


void MetaEdit::setEmptyText() {
	Q_ASSERT(mDocument->commentState()!=Document::NONE);
	if (mDocument->commentState()==Document::WRITABLE) {
		if (mCommentEdit->hasFocus()) {
			setComment("");
		} else {
			setMessage(i18n("Type here to add a comment to this image."));
		}
	} else {
		setMessage(i18n("No comment available."));
	}
}


/**
 * Use mCommentEdit to show the comment and let the user edit it
 */
void MetaEdit::setComment(const TQString& comment) {
	Q_ASSERT(mDocument->commentState()!=Document::NONE);
	mCommentEdit->setTextFormat(TQTextEdit::PlainText);
	mCommentEdit->setReadOnly(mDocument->commentState()==Document::READ_ONLY);
	mCommentEdit->setText(comment);
}


/**
 * Use mCommentEdit to display a read-only message
 */
void MetaEdit::setMessage(const TQString& msg) {
	mCommentEdit->setTextFormat(TQTextEdit::RichText);
	mCommentEdit->setReadOnly(true);
	mCommentEdit->setText(TQString("<i>%1</i>").arg(msg));
}

} // namespace
